<?php

namespace App\Util;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Monolog\Logger;

class TokenManager
{
    private const TOKEN_FILE = __DIR__ . '/../../zoho_token.json';
    private Logger $log;
    private array $tokenData = [];

    public function __construct(Logger $log)
    {
        $this->log = $log;
    }

    public function getAccessToken(): string
    {
        $this->loadTokenFromFile();

        if ($this->isTokenValid()) {
            $this->log->info("Używanie tokena Zoho z cache. Ważny do: " . date('Y-m-d H:i:s', $this->tokenData['expires_at']));
            return $this->tokenData['access_token'];
        }

        $this->log->warning("Token Zoho jest nieważny, nie istnieje lub wkrótce wygaśnie. Odświeżanie...");
        $this->refreshNewToken();
        
        return $this->tokenData['access_token'];
    }

    /**
     * NOWA METODA: Pozwala na wymuszenie odświeżenia tokena,
     * ignorując jego datę ważności w cache.
     */
    public function forceRefreshAndGetNewToken(): string
    {
        $this->log->warning("Wymuszone odświeżenie tokena z powodu błędu autoryzacji (401).");
        $this->refreshNewToken();
        return $this->tokenData['access_token'];
    }

    private function loadTokenFromFile(): void
    {
        if (file_exists(self::TOKEN_FILE)) {
            $this->tokenData = json_decode(file_get_contents(self::TOKEN_FILE), true) ?? [];
        }
    }

    private function saveTokenToFile(): void
    {
        file_put_contents(self::TOKEN_FILE, json_encode($this->tokenData, JSON_PRETTY_PRINT));
    }

    private function isTokenValid(): bool
    {
        if (empty($this->tokenData['access_token']) || empty($this->tokenData['expires_at'])) {
            return false;
        }
        return (time() < ($this->tokenData['expires_at'] - 60));
    }

    private function refreshNewToken(): void
    {
        $this->log->info('Odświeżanie tokena dostępu Zoho z serwera...');
        $authUrl = "https://accounts.zoho.eu/oauth/v2/token";
        
        try {
            $client = new Client();
            $response = $client->post($authUrl, [
                'form_params' => [
                    'refresh_token' => $_ENV['ZOHO_API_REFRESH_TOKEN'],
                    'client_id'     => $_ENV['ZOHO_API_CLIENT_ID'],
                    'client_secret' => $_ENV['ZOHO_API_CLIENT_SECRET'],
                    'grant_type'    => 'refresh_token',
                ]
            ]);
            
            $data = json_decode($response->getBody()->getContents(), true);

            if (isset($data['access_token'])) {
                $this->tokenData['access_token'] = $data['access_token'];
                $this->tokenData['expires_at'] = time() + $data['expires_in'];
                $this->saveTokenToFile();
                $this->log->info('Pomyślnie odświeżono i zapisano nowy token Zoho.');
            } else {
                throw new \Exception('Odpowiedź z Zoho nie zawiera access_token. Odpowiedź: ' . json_encode($data));
            }
        } catch (RequestException $e) {
            $responseBody = $e->hasResponse() ? $e->getResponse()->getBody()->getContents() : $e->getMessage();
            $this->log->error('KRYTYCZNY BŁĄD odświeżania tokena Zoho: ' . $responseBody);
            die('Nie można uzyskać tokena Zoho. Zatrzymywanie skryptu.');
        }
    }
}