<?php

namespace App\Api;

use App\Util\TokenManager;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Monolog\Logger;

class ZohoApiClient
{
    private Client $client;
    private Logger $log;
    private TokenManager $tokenManager;

    public function __construct(Logger $log, TokenManager $tokenManager)
    {
        $this->client = new Client(['timeout' => 60.0]);
        $this->log = $log;
        $this->tokenManager = $tokenManager;
    }
    
    // ... (metody publiczne, jak fetchAllRecords, upsertRecords etc. pozostają bez zmian) ...
    public function fetchSingleRecord(string $module, string $recordId): ?array
    {
        $response = $this->get("{$_ENV['ZOHO_API_BASE_URI']}/crm/v2/{$module}/{$recordId}", []);
        return $response['data'][0] ?? null;
    }
    
    public function fetchAllRecords(string $module): array
    {
        $allRecords = [];
        $page = 1;
        $moreRecords = true;

        while ($moreRecords) {
            // ZMIANA: Usunięto dodawanie nagłówka autoryzacji tutaj, zajmie się tym metoda get()
            $response = $this->get("{$_ENV['ZOHO_API_BASE_URI']}/crm/v2/{$module}", [
                'query' => ['page' => $page, 'per_page' => 200]
            ]);
            
            $data = $response['data'] ?? [];
            if (!empty($data)) {
                $allRecords = array_merge($allRecords, $data);
            }

            $info = $response['info'] ?? [];
            if (isset($info['more_records']) && $info['more_records']) {
                $page++;
            } else {
                $moreRecords = false;
            }
        }
        return $allRecords;
    }
    
    public function upsertRecords(string $module, array $records, string $uniqueField): void
    {
        if (empty($records)) {
            $this->log->info("Brak rekordów do wysłania do modułu {$module}.");
            return;
        }

        $enableTriggers = filter_var($_ENV['ZOHO_WORKFLOW_TRIGGERS_ENABLED'] ?? false, FILTER_VALIDATE_BOOLEAN);
        $headers = []; // Nagłówek autoryzacji zostanie dodany przez metodę post()

        if (!$enableTriggers) {
            $headers['X-ZOHO-CRM-API-TRIGGER'] = '[]';
            $this->log->info("Operacja na module {$module} z WYŁĄCZONYMI triggerami (zgodnie z .env).");
        } else {
            $this->log->info("Operacja na module {$module} z WŁĄCZONYMI triggerami (zgodnie z .env).");
        }

        $chunks = array_chunk($records, 100);
        foreach ($chunks as $index => $chunk) {
            $this->log->info("Wysyłanie paczki " . ($index + 1) . "/" . count($chunks) . " do modułu {$module}...");
            $response = $this->post("{$_ENV['ZOHO_API_BASE_URI']}/crm/v2/{$module}/upsert", [
                'headers' => $headers,
                'json' => ['data' => $chunk, 'duplicate_check_fields' => [$uniqueField]]
            ]);
            
            if ($response !== null && isset($response['data']) && is_array($response['data'])) {
                foreach ($response['data'] as $key => $result) {
                    if (isset($result['status']) && $result['status'] === 'error') {
                        $failedRecordData = $chunk[$key] ?? 'Nie udało się zidentyfikować rekordu';
                        $this->log->error("BŁĄD ZAPISU W ZOHO: Rekord został odrzucony przez API.", [
                            'Wysłane dane' => $failedRecordData,
                            'Odpowiedź błędu z Zoho' => $result
                        ]);
                    }
                }
            }
        }
    }

    public function runCOQLQuery(string $query): array
    {
        $this->log->info("Wykonywanie zapytania COQL: {$query}");
        $response = $this->post("{$_ENV['ZOHO_API_BASE_URI']}/crm/v2/coql", [
            'json' => ['select_query' => $query]
        ]);
        return $response['data'] ?? [];
    }

    /**
     * NOWA PRYWATNA METODA POMOCNICZA z logiką ponawiania
     */
    private function executeRequest(string $method, string $url, array $options, bool $isRetry = false): ?array
    {
        // Upewnij się, że tablica headers istnieje
        if (!isset($options['headers'])) {
            $options['headers'] = [];
        }

        // Dodaj aktualny token do nagłówków
        $options['headers']['Authorization'] = "Zoho-oauthtoken {$this->tokenManager->getAccessToken()}";

        try {
            $response = $this->client->request($method, $url, $options);
            return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            // Sprawdź czy to błąd 401 i czy jeszcze nie ponawialiśmy próby
            if (!$isRetry && $e->hasResponse() && $e->getResponse()->getStatusCode() === 401) {
                $this->log->warning("Otrzymano błąd 401 (INVALID_TOKEN). Wymuszam odświeżenie tokena i ponawiam zapytanie...");
                
                // Wymuś odświeżenie tokena
                $this->tokenManager->forceRefreshAndGetNewToken();
                
                // Ponów zapytanie (rekurencyjnie), oznaczając, że to już jest ponowiona próba
                return $this->executeRequest($method, $url, $options, true);
            }

            // Dla wszystkich innych błędów lub dla nieudanej ponowionej próby, zaloguj i zwróć null
            $errorContext = $e->hasResponse() ? $e->getResponse()->getBody()->getContents() : 'Brak odpowiedzi serwera.';
            $this->log->error("Błąd API Zoho [{$method} {$url}]: " . $e->getMessage());
            $this->log->error("Odpowiedź serwera Zoho: " . $errorContext);
            return null;
        }
    }
    
    // Metody get() i post() teraz używają nowej, inteligentnej metody executeRequest
    public function get(string $url, array $options): ?array 
    {
        return $this->executeRequest('GET', $url, $options);
    }
    
    public function post(string $url, array $options): ?array 
    {
        return $this->executeRequest('POST', $url, $options);
    }

    

    // ... na końcu klasy ZohoApiClient, przed ostatnim zamykającym nawiasem klamrowym } ...

    /**
     * NOWA METODA: Wysyła e-mail przez API Zoho używając Notification_Groups.
     */
    public function sendEmail(array $recipientsData, string $subject, string $htmlContent): bool
    {
        $this->log->info("Przygotowywanie e-maila o temacie '{$subject}'...");

        $toList = [];
        foreach ($recipientsData as $recipient) {
            $toList[] = ['email' => $recipient['email']];
        }

        $payload = [
            'send_mail' => [
                'from' => [
                    'email' => $_ENV['SENDER_EMAIL'],
                    'name' => $_ENV['SENDER_NAME']
                ],
                'to' => $toList,
                'subject' => $subject,
                'content' => $htmlContent,
                'mail_format' => 'html'
            ]
        ];
        
        // Użyj tego samego ID grupy notyfikacji, co w Deluge
        $url = "{$_ENV['ZOHO_API_BASE_URI']}/crm/v7/Notification_Groups/612527000010433878/actions/send_mail";

        $response = $this->post($url, ['json' => $payload]);

        if ($response !== null && isset($response['status']) && $response['status'] === 'success') {
            $this->log->info("E-mail został pomyślnie wysłany przez API Zoho.");
            return true;
        }

        $this->log->error("Nie udało się wysłać e-maila przez API Zoho.", ['response' => $response]);
        return false;
    }
}
