<?php
// Plik: src/Api/GptClient.php

namespace App\Api;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Monolog\Logger;

class GptClient
{
    private Client $client;
    private Logger $log;
    private string $apiKey;

    public function __construct(Logger $log, string $apiKey)
    {
        $this->log = $log;
        $this->apiKey = $apiKey;

        // Inicjalizujemy klienta Guzzle z domyślnymi ustawieniami dla OpenAI
        $this->client = new Client([
            'base_uri' => 'https://api.openai.com/v1/',
            'timeout'  => 300.0, // Hardcoded 5 minut na odpowiedź z API
            'headers' => [
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type'  => 'application/json',
            ]
        ]);
    }

    /**
     * Generuje odpowiedź z modelu czatu OpenAI.
     *
     * @param string $promptText Tekst promptu od użytkownika.
     * @param string $model Model do użycia (np. 'gpt-4o-mini').
     * @param float $temperature "Kreatywność" odpowiedzi (0.0 - 2.0).
     * @param array $options Dodatkowe opcje API, np. ['max_tokens' => 2000].
     * @return string|null Wygenerowany tekst lub null w przypadku błędu.
     */
    public function generateChatCompletion(string $promptText, string $model = 'gpt-4o-mini', float $temperature = 0.3, array $options = []): ?string
    {
        $this->log->info("Przygotowywanie zapytania do GPT dla modelu {$model} z temperaturą {$temperature}.");
        
        $apiBody = [
            'model' => $model,
            'messages' => [
                ['role' => 'user', 'content' => $promptText]
            ],
            'temperature' => $temperature,
        ];

        $apiBody = array_merge($apiBody, $options);

        try {
            $this->log->info('Wysyłanie zapytania do OpenAI API...');
            $response = $this->client->post('chat/completions', ['json' => $apiBody]);
            $responseData = json_decode($response->getBody()->getContents(), true);

            $content = $responseData['choices'][0]['message']['content'] ?? null;

            if ($content) {
                $this->log->info('Pomyślnie otrzymano odpowiedź z OpenAI.');
                return trim($content);
            } else {
                $this->log->warning('Odpowiedź z OpenAI nie zawierała oczekiwanej treści.', ['response' => $responseData]);
                return null;
            }

        } catch (RequestException $e) {
            $errorContext = $e->hasResponse() ? $e->getResponse()->getBody()->getContents() : 'Brak odpowiedzi serwera.';
            $this->log->error("Błąd API OpenAI: " . $e->getMessage(), ['response' => $errorContext]);
            return null;
        }
    }
}