<?php
// Plik: pb_worker.php (Wersja Produkcyjna)

/** @var \App\Api\ZohoApiClient $zohoApiClient */
/** @var \Monolog\Logger $logger */
/** @var \App\Api\GptClient $gptClient */
[$zohoApiClient, $logger, $gptClient] = require_once __DIR__ . '/pb_bootstrap.php';


$lockFile = __DIR__ . '/pb_worker.lock';
$lockHandle = fopen($lockFile, 'w');
if (!$lockHandle || !flock($lockHandle, LOCK_EX | LOCK_NB)) {
    $logger->info('Worker is already running. Exiting.');
    exit;
}

$taskFile = null;
try {
    $queueDir = __DIR__ . '/queue/';
    $tasks = glob($queueDir . 'pb_*.json');
    if (empty($tasks)) {
        $logger->info('No tasks in queue.');
        exit;
    }

    sort($tasks);
    $taskFile = $tasks[0];
    
    $logger->info("Processing task: " . basename($taskFile));

    $payload = file_get_contents($taskFile);
    $taskData = json_decode($payload, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception("Invalid JSON in task file.");
    }

    $recipients = $taskData['recipients'] ?? [];
    if (empty($recipients)) {
        throw new Exception("No recipients found in the task data.");
    }
    
    $toList = [];
    $RecName = "";
    foreach ($recipients as $recipient) 
    {
        $toList[] = ['email' => $recipient['email']];
        $RecName = $recipient['name'];
    }

    $logger->info(" RecName= ".  $RecName);

    // === Konfiguracja === NEWSY z RSS
    $rssUrl = 'https://www.polsatnews.pl/rss/polska.xml';
    $interval = 9;
    $finalString = "";

    $xmlData = @file_get_contents($rssUrl);

    if ($xmlData === false) {
        $finalString = "Błąd: Nie udało się pobrać danych z serwera RSS.";
        $logger->warning($finalString);
    } else {
        $rss = @simplexml_load_string($xmlData);
        if ($rss === false) {
            $finalString = "Błąd: Pobrane dane nie są poprawnym plikiem XML.";
            $logger->warning($finalString);
        } else {
            $itemCount = 0;
            foreach ($rss->channel->item as $item) {
                $itemCount++;
                if ($itemCount % $interval === 0) {
                    $pubDate = date('Y-m-d H:i:s', strtotime($item->pubDate));
                    $title = (string) $item->title;
                    $description = trim(strip_tags($item->description));
                    $finalString .= "Data publikacji: {$pubDate}\n";
                    $finalString .= "Tytuł: {$title}\n";
                    $finalString .= "Opis: {$description}\n";
                    $finalString .= "-----------------------\n\n";
                }
            }
        }
    }
    $gptNewsResponseText = $finalString;

    // Logika specyficzna dla roli
    $role = $taskData['recipient_role'] ?? 'PM';
    $finalEmailHtml = ''; // Inicjalizacja zmiennej

    if($role == 'PM')
    {
        $logger->info("Wczytywanie szablonu promptu i e-maila dla roli: PM");
        $pmPromptTemplate = file_get_contents(__DIR__ . '/pm_prompt.txt');
        $alertEmailTemplate = file_get_contents(__DIR__ . '/alert_template.html');

        $alertsText = $taskData['alerts_text'];
        $finalGptPrompt = "Piszesz do PM: " . $RecName ." \n\n". $pmPromptTemplate . $alertEmailTemplate . "[NEWSY ZE ŚWIATA I POLSKI:]   " . $gptNewsResponseText."[ALERTY UZYTKOWNIKA]". $alertsText;

        $gptResponseText = $gptClient->generateChatCompletion(
            promptText: $finalGptPrompt,
            model: 'gpt-4o', // Sugestia: użyj 'gpt-4o' lub 'gpt-4o-mini' dla lepszej wydajności i kosztów
            temperature: 0.8
        );

        if ($gptResponseText === null) {
            throw new Exception("GPT API call failed or returned empty content for PM.");
        }
        
        $logger->info("Otrzymano i przetworzono odpowiedź z GPT dla PM.");
        $finalEmailHtml = $gptResponseText;
    }
    
    if($role == 'ADMIN')
    {
        $logger->info("Wczytywanie szablonu promptu i e-maila dla roli: ADMIN");
        $pmPromptTemplate = file_get_contents(__DIR__ . '/admin_prompt.txt');
        $alertEmailTemplate = file_get_contents(__DIR__ . '/alert_template.html');

        $alertsText = $taskData['alerts_text'];
        $finalGptPrompt = "Piszesz do samego ADMINA! \n\n". $pmPromptTemplate . $alertEmailTemplate . "[NEWSY ZE ŚWIATA I POLSKI:]   " . $gptNewsResponseText."[ALERTY UZYTKOWNIKÓW]". $alertsText;
        
        $gptResponseText = $gptClient->generateChatCompletion(
            promptText: $finalGptPrompt,
            model: 'gpt-4o', // Sugestia: użyj 'gpt-4o' lub 'gpt-4o-mini'
            temperature: 0.8
        );

        if ($gptResponseText === null) {
            throw new Exception("GPT API call failed or returned empty content for ADMIN.");
        }
        
        $logger->info("Otrzymano i przetworzono odpowiedź z GPT dla ADMIN.");
        $finalEmailHtml = $gptResponseText;
    }
    
    // --- LOGIKA WYSYŁKI E-MAILA ---        
    $subject = "Powiadomienia Profit BOT dla {$role} - " . date('Y-m-d');
    
    $emailObject = [
        'from' => [ 'email' => $_ENV['SENDER_EMAIL'], 'name' => $_ENV['SENDER_NAME'] ],
        'to' => $toList,
        'subject' => $subject,
        'content' => $finalEmailHtml,
        'mail_format' => 'html',
        'org_email' => true
    ];

    $emailApiPayload = ['data' => [ $emailObject ]];
    $logger->info('Wysyłanie do Zoho API następującego payloadu:', ['payload_size' => strlen(json_encode($emailApiPayload))]);

    $sendMailUrl = "{$_ENV['ZOHO_API_BASE_URI']}/crm/v7/Notification_Groups/612527000010433878/actions/send_mail";
    $response = $zohoApiClient->post($sendMailUrl, ['json' => $emailApiPayload]);

    if (isset($response['data'][0]['status']) && $response['data'][0]['status'] === 'success') {
        $logger->info("E-mail został pomyślnie wysłany przez API Zoho.");
    } else {
        throw new Exception("Zoho API failed to send the email.");
    }

    $logger->info("Task " . basename($taskFile) . " processed successfully.");
    
    // ODKOMENTOWANE: Usuwamy plik po pomyślnym przetworzeniu
    unlink($taskFile); 

} catch (Exception $e) {
    $errorMessage = "Failed to process task " . basename($taskFile ?? 'N/A') . ": " . $e->getMessage();
    $logger->error($errorMessage, ['exception' => $e]);
    
    // ODKOMENTOWANE: Wysyłamy e-mail do admina w razie błędu
    mail($_ENV['ADMIN_EMAIL'], "Profit BOT Worker Error", "Error: " . $e->getMessage());
    
    // ODKOMENTOWANE: Przenosimy plik z błędem do folderu /error
    if (isset($taskFile) && file_exists($taskFile)) {
        $errorDir = __DIR__ . '/error/';
        if (!is_dir($errorDir)) { mkdir($errorDir); }
        rename($taskFile, $errorDir . basename($taskFile));
    }
} finally {
    flock($lockHandle, LOCK_UN);
    fclose($lockHandle);
    @unlink($lockFile);
}